/*
 * @(#)AMView.java  1.0  2006-12-06
 *
 * Copyright (c) 2006 Lucerne University of Applied Sciences and Arts (HSLU)
 * Zentralstrasse 18, Postfach 2858, CH-6002 Lucerne, Switzerland
 * All rights reserved.
 *
 * The copyright of this software is owned by the Lucerne University of Applied 
 * Sciences and Arts (HSLU). You may not use, copy or modify this software, 
 * except in accordance with the license agreement you entered into with HSLU. 
 * For details see accompanying license terms. 
 */
package ch.hslu.cm.am;

import ch.hslu.cm.app.action.Arrangeable;
import ch.hslu.cm.am.diagram.*;
import ch.hslu.cm.am.model.*;
import ch.randelshofer.io.*;
import ch.randelshofer.util.*;
import java.awt.*;
import java.awt.event.*;
import java.beans.*;
import java.io.*;
import java.text.MessageFormat;
import java.util.*;
import java.util.prefs.*;
import java.util.zip.*;
import javax.swing.*;
import javax.swing.border.*;
import java.lang.reflect.*;
import java.net.URI;
import org.jhotdraw.app.*;
import org.jhotdraw.app.action.edit.UndoAction;
import org.jhotdraw.draw.*;
import org.jhotdraw.draw.event.ToolAdapter;
import org.jhotdraw.draw.event.ToolEvent;
import org.jhotdraw.draw.event.ToolListener;
import org.jhotdraw.draw.io.DOMStorableInputOutputFormat;
import org.jhotdraw.draw.io.ImageOutputFormat;
import org.jhotdraw.draw.io.InputFormat;
import org.jhotdraw.draw.io.OutputFormat;
import org.jhotdraw.draw.tool.Tool;
import org.jhotdraw.gui.JFileURIChooser;
import org.jhotdraw.gui.URIChooser;
import org.jhotdraw.gui.filechooser.ExtensionFileFilter;
import org.jhotdraw.undo.UndoRedoManager;
import org.jhotdraw.util.ResourceBundleUtil;
import org.jhotdraw.xml.NanoXMLDOMInput;
import org.jhotdraw.xml.NanoXMLDOMOutput;
/**
 * Activity Modeling View.
 *
 * @author Werner Randelshofer, Florian Padrun
 * @version 1.0 2006-12-06 Created.
 */
public class AMView extends AbstractView
        implements Arrangeable {
    private Arrangeable.Arrangement arrangement;
    private JSplitPane splitPane;
    private JTabbedPane splittedProblemPanel, splittedSolutionPanel;
    private ResourceBundleUtil labels;
    private Preferences prefs;
    private UndoRedoManager undo;
    private DrawingEditor editor;
    private AMFactory domFactory;
    private boolean isOptionsVisible = true;
    private GridConstrainer visibleConstrainer = new GridConstrainer(10, 10);
    private GridConstrainer invisibleConstrainer = new GridConstrainer(1, 1);
    
    private class ToolButtonListener implements ItemListener {
        private Tool tool;
        public ToolButtonListener(Tool t) {
            this.tool = t;
        }
        public void itemStateChanged(ItemEvent evt) {
            if (evt.getStateChange() == ItemEvent.SELECTED) {
                editor.setTool(tool);
            }
        }
    }
    
    /** Creates new form. */
    public AMView() {
    }
    public void init() {
        super.init();
        labels = ResourceBundleUtil.getBundle("ch.hslu.cm.am.Labels");
        prefs = Preferences.userNodeForPackage(getClass());
        undo = new UndoRedoManager();
        
        initComponents();
        initActions();
        
        problemDrawingView.setDrawing(new ActivityDiagram());
        solutionDrawingView.setDrawing(new ActivityDiagram());
        
        DOMStorableInputOutputFormat domIOF = new DOMStorableInputOutputFormat(new AMFactory());
        LinkedList<InputFormat> inputFormats = new LinkedList<InputFormat>();
        inputFormats.add(domIOF);
        LinkedList<OutputFormat> outputFormats = new LinkedList<OutputFormat>();
        outputFormats.add(domIOF);
        outputFormats.add(new ImageOutputFormat());
        problemDrawingView.getDrawing().setInputFormats(inputFormats);
        solutionDrawingView.getDrawing().setInputFormats(inputFormats);
        problemDrawingView.getDrawing().setOutputFormats(outputFormats);
        solutionDrawingView.getDrawing().setOutputFormats(outputFormats);
        
        problemDrawingView.setEmptyDrawingMessage(labels.getString(
                "emptyProblemMessage"));
        solutionDrawingView.setEmptyDrawingMessage(labels.getString(
                "emptySolutionMessage"));
        
        splitPane = new JSplitPane();
        splitPane.setBorder(new EmptyBorder(0,0,0,0));
        splitPane.add(splittedProblemPanel = new JTabbedPane(), splitPane.LEFT);
        splitPane.add(splittedSolutionPanel = new JTabbedPane(), splitPane.RIGHT);
        splittedProblemPanel.setTabPlacement(JTabbedPane.BOTTOM);
        splittedSolutionPanel.setTabPlacement(JTabbedPane.BOTTOM);
        
        // Get/Store divider location from/to preferences
        splitPane.setDividerLocation(prefs.getInt("project.dividerLocation", 100));
        splitPane.addPropertyChangeListener(new PropertyChangeListener() {
            public void propertyChange(PropertyChangeEvent evt) {
                if (evt.getPropertyName().equals("dividerLocation")) {
                    prefs.putInt("project.dividerLocation", 
                            splitPane.getDividerLocation());
                }
            }
        });
        mainSplitPane.setDividerLocation(prefs.getInt("project.mainDividerLocation", 400));
        mainSplitPane.addPropertyChangeListener(new PropertyChangeListener() {
            public void propertyChange(PropertyChangeEvent evt) {
                if (evt.getPropertyName().equals("dividerLocation")) {
                    prefs.putInt("project.mainDividerLocation",
                            mainSplitPane.getDividerLocation());
                }
            }
        });
        
        
        // Get arrangement from preferences
        setArrangement(Arrangeable.Arrangement.valueOf(
                prefs.get("project.arrangement", 
                Arrangeable.Arrangement.CASCADE.toString())
                ));
        
        setDrawingEditor(new DefaultDrawingEditor());
        
        setOptionsVisible(prefs.getBoolean("project.optionsVisible", true));
        setGridVisible(prefs.getBoolean("project.gridVisible", false));
        setScaleFactor(prefs.getDouble("project.scaleFactor", 1d));
        
        splittedProblemPanel.setTabLayoutPolicy(JTabbedPane.WRAP_TAB_LAYOUT);
        splittedSolutionPanel.setTabLayoutPolicy(JTabbedPane.WRAP_TAB_LAYOUT);
        tabbedPane.setTabLayoutPolicy(JTabbedPane.WRAP_TAB_LAYOUT);
        mainSplitPane.putClientProperty("Quaqua.SplitPane.style","bar");
        splitPane.putClientProperty("Quaqua.SplitPane.style","bar");
        Font smallSystemFont = UIManager.getFont("SmallSystemFont");
        if (smallSystemFont !=  null) {
            splittedProblemPanel.setFont(smallSystemFont);
            splittedSolutionPanel.setFont(smallSystemFont);
            tabbedPane.setFont(smallSystemFont);
        }
        splittedProblemPanel.putClientProperty("Quaqua.Component.visualMargin",
                new Insets(0,0,3,0));
        splittedSolutionPanel.putClientProperty("Quaqua.Component.visualMargin",
                new Insets(0,0,3,0));
        tabbedPane.putClientProperty("Quaqua.Component.visualMargin",
                new Insets(0,0,3,0));
        problemScrollPane.putClientProperty("Quaqua.TabbedPaneChild.contentInsets",
                new Insets(0,0,0,0));
        solutionScrollPane.putClientProperty("Quaqua.TabbedPaneChild.contentInsets",
                new Insets(0,0,0,0));
        
        undo.addPropertyChangeListener(new PropertyChangeListener() {
            public void propertyChange(PropertyChangeEvent evt) {
                setHasUnsavedChanges(undo.hasSignificantEdits());
            }
        });
        
        problemDrawingView.getDrawing().addUndoableEditListener(undo);
        solutionDrawingView.getDrawing().addUndoableEditListener(undo);
        optionsPanel.addUndoableEditListener(undo);
    }
    
    /**
     * By convention, this method is never invoked on the AWT event dispatcher
     * thread.
     */
    public void writeHTML(File f, javax.swing.filechooser.FileFilter filter,
            Component accessory) throws IOException {
        // XXX - Implement me for JHotDraw 7.4.
        prefs.put("projectExportFile", f.getPath());
        
        if (filter instanceof ExtensionFileFilter) {
            ExtensionFileFilter eff = (ExtensionFileFilter) filter;
            if (eff.getExtensions().contains("html")) {
                String baseName = f.getName();
                File baseDir = f.getParentFile();
                
                if (baseName.toLowerCase().endsWith(".html")) {
                    baseName = baseName.substring(0, baseName.length() - 5);
                } else  if (baseName.toLowerCase().endsWith(".htm")) {
                    baseName = baseName.substring(0, baseName.length() - 4);
                }
                
                // Write libraries
                ZipInputStream zin = null;
                try {
                    zin = new ZipInputStream(getClass().getResourceAsStream(
                            "/htmlexport.zip"));
                    ZipFiles.unzip(zin, baseDir);
                } finally {
                    if (zin != null) {
                        zin.close();
                    }
                }
                
                // Write HTML page
                InputStream in = null;
                String html = null;
                try {
                    in = getClass().getResourceAsStream(
                            (problemDrawingView.getDrawing().getChildCount()
                            == 0 &&
                            solutionDrawingView.getDrawing().getChildCount()
                            == 0) ?
                                "/ch/hslu/cm/prototype_cmtext_sco.html" :
                                "/ch/hslu/cm/prototype_cmapplet_sco.html"
                            );
                    String htmlPrototype = Streams.toString(in);
                    html = MessageFormat.format(
                            htmlPrototype,
                            optionsPanel.getChapterText(),
                            optionsPanel.getTitleText(),
                            optionsPanel.getTitleText(),
                            optionsPanel.getInstructionsTextHTML(),
                            optionsPanel.getInstructionLanguage().getLanguage(),
                            baseName+".xml",
                            "ch.hslu.cm.am.AMLiveConnectApplet",
                            "cmamapplet.jar"
                            );
                } finally {
                    if (in != null) {
                        in.close();
                    }
                }
                OutputStreamWriter w = null;
                try {
                    w = new OutputStreamWriter(new FileOutputStream(new File(
                            baseDir, baseName+".html")), "UTF-8");
                    w.write(html);
                } finally {
                    if (w != null) {
                        w.close();
                    }
                }
                
                // Write project document
                writeXML(new File(baseDir, baseName+".xml").toURI(),null);
                
                return;
            }
        }
        throw new IOException(labels.getFormatted("exportCantExportFileAs",f,
                filter.getDescription()));
    }
    
    private void initActions() {
        // XXX - Implement me for JHotDraw 7.4.
        /*
        putAction(UndoAction.ID, undo.getUndoAction());
        putAction(RedoAction.ID, undo.getRedoAction());
        putAction(SelectAllAction.ID, new SelectAllAction());
        putAction(DuplicateAction.ID, new DuplicateAction());
        AbstractAction toggleOptionsAction = new AbstractAction(labels.getString(
                "workAssignment")) {
            public void actionPerformed(ActionEvent evt) {
                setOptionsVisible(! isOptionsVisible());
                putValue(Actions.SELECTED_KEY, isOptionsVisible());
            }
        };
        toggleOptionsAction.putValue(Actions.SELECTED_KEY, isOptionsVisible());
        putAction("toggleOptionsPanel", toggleOptionsAction);
         *
         */
    }
    
    public ActivityModel getProblemModel() {
        return ((ActivityDiagram) 
        problemDrawingView.getDrawing()).getActivityModel();
    }
    public ActivityModel getSolutionModel() {
        return ((ActivityDiagram) 
        solutionDrawingView.getDrawing()).getActivityModel();
    }
    
    @Override
    public void write(URI f, URIChooser chooser) throws IOException {
        if (chooser instanceof JFileURIChooser) {
            JFileURIChooser jfuri=(JFileURIChooser)chooser;
            if (jfuri.getFileFilter() instanceof ExtensionFileFilter) {
                ExtensionFileFilter eff=(ExtensionFileFilter)jfuri.getFileFilter();
                if (eff.getExtensions().contains("html")) {
                    writeHTML(new File(f),eff,jfuri.getAccessory());
                    return;
                }
            }
        }
        writeXML(f,chooser);
        }

    public void writeXML(URI f, URIChooser chooser) throws IOException {
        PrintWriter out = null;
        try {
            out = new PrintWriter(new OutputStreamWriter(
                    new BufferedOutputStream(new FileOutputStream(new File(f))), "UTF8"));
            NanoXMLDOMOutput domo = new NanoXMLDOMOutput(new AMFactory());
            domo.openElement("ConceptModelerAM");
            domo.openElement("Chapter");
            domo.writeObject(optionsPanel.getChapterText());
            domo.closeElement();
            domo.openElement("Title");
            domo.writeObject(optionsPanel.getTitleText());
            domo.closeElement();
            domo.openElement("Instructions");
            domo.writeObject(optionsPanel.getInstructionsText());
            domo.closeElement();
            domo.openElement("Language");
            domo.writeObject(optionsPanel.getInstructionLanguage().getLanguage());
            domo.closeElement();
            domo.openElement("Problem");
            ActivityDiagram diagram;
            diagram = (ActivityDiagram) problemDrawingView.getDrawing();
            domo.writeObject(diagram.getSimulation());
            domo.writeObject(diagram);
            domo.closeElement();
            domo.openElement("Solution");
            diagram = (ActivityDiagram) solutionDrawingView.getDrawing();
            domo.writeObject(diagram.getSimulation());
            domo.writeObject(diagram);
            domo.closeElement();
            domo.closeElement();
            domo.print(out);
        } finally {
            out.close();
            //if (out != null) try { out.close(); } catch (IOException e) {};
        }
        
    }
    
    @Override
    public void read(URI f, URIChooser chooser) throws IOException {
        InputStream in = null;
        try {
            in = new BufferedInputStream(new FileInputStream(new File(f)));
            NanoXMLDOMInput domi = new NanoXMLDOMInput(new AMFactory(), in);
            if (domi.getElementCount("ConceptModelerAM") == 0) {
                new IOException("Couldn't find root element named: " +
                        "ConceptModelerAM");
            }
            domi.openElement("ConceptModelerAM");
            
            final String chapterText;
            if (domi.getElementCount("Chapter") > 0) {
                domi.openElement("Chapter");
                chapterText = (String) domi.readObject();
                domi.closeElement();
            } else {
                chapterText = "";
            }
            final String titleText;
            if (domi.getElementCount("Title") > 0) {
                domi.openElement("Title");
                titleText = (String) domi.readObject();
                domi.closeElement();
            } else {
                titleText = "";
            }
            final String instructionsText;
            if (domi.getElementCount("Instructions") > 0) {
                domi.openElement("Instructions");
                instructionsText = (String) domi.readObject();
                domi.closeElement();
            } else {
                instructionsText = "";
            }
            final String language;
            if (domi.getElementCount("Language") > 0) {
                domi.openElement("Language");
                language = (String) domi.readObject();
                domi.closeElement();
            } else {
                language = "en";
            }
            final ActivityDiagram pDiagram;
            if (domi.getElementCount("Problem") > 0) {
                domi.openElement("Problem");
                ActivityModel pSimulation = (ActivityModel) domi.readObject(0);
                pDiagram = (ActivityDiagram) domi.readObject(1);
                domi.closeElement();
            } else {
                pDiagram = new ActivityDiagram();
            }
            final ActivityDiagram sDiagram;
            if (domi.getElementCount("Solution") > 0) {
                domi.openElement("Solution");
                ActivityModel sSimulation = (ActivityModel) domi.readObject(0);
                sDiagram = (ActivityDiagram) domi.readObject(1);
                domi.closeElement();
                domi.closeElement();
            } else {
                sDiagram = new ActivityDiagram();
            }
            
            SwingUtilities.invokeAndWait(new Runnable() { public void run() {
                problemDrawingView.getDrawing().removeUndoableEditListener(undo);
                solutionDrawingView.getDrawing().removeUndoableEditListener(undo);
                problemDrawingView.setDrawing(pDiagram);
                solutionDrawingView.setDrawing(sDiagram);
                problemDrawingView.getDrawing().addUndoableEditListener(undo);
                solutionDrawingView.getDrawing().addUndoableEditListener(undo);
                optionsPanel.setChapterText(chapterText);
                optionsPanel.setTitleText(titleText);
                optionsPanel.setInstructionsText(instructionsText);
                optionsPanel.setInstructionLanguage(new Locale(language));
                undo.discardAllEdits();
            }});
        } catch (InterruptedException e) {
            InternalError error = new InternalError();
            error.initCause(e);
            throw error;
        } catch (InvocationTargetException e) {
            InternalError error = new InternalError();
            error.initCause(e);
            throw error;
        } catch (Throwable e) {
            IOException error = new IOException("Error: "+e.getMessage());
            error.initCause(e);
            throw error;
        } finally {
            if (in != null) try { in.close(); } catch (IOException e) {};
        }
    }
    
    public void setDrawingEditor(DrawingEditor newValue) {
        if (editor != null) {
            editor.remove(problemDrawingView);
            editor.remove(solutionDrawingView);
        }
        editor = newValue;
        if (editor != null) {
            editor.add(problemDrawingView);
            editor.add(solutionDrawingView);
        }
    }
    
    public DrawingEditor getDrawingEditor() {
        return editor;
    }
    
    public void setOptionsVisible(boolean newValue) {
        if (isOptionsVisible != newValue) {
            boolean oldValue = isOptionsVisible;
            isOptionsVisible = newValue;
            
            removeAll();
            if (isOptionsVisible) {
                add(mainSplitPane);
                mainRightPanel.add(mainPanel);
                optionsPanel.invalidate();
                mainLeftPanel.invalidate();
                mainRightPanel.invalidate();
                
            } else {
                add(mainPanel);
            }
            revalidate();
            repaint();
            
            firePropertyChange("optionsVisible", oldValue, newValue);
            prefs.putBoolean("project.optionsVisible", newValue);
        }
    }
    public boolean isOptionsVisible() {
        return isOptionsVisible;
    }
    public void setGridVisible(boolean newValue) {
        boolean oldValue = isGridVisible();
        problemDrawingView.setConstrainerVisible(newValue);
        solutionDrawingView.setConstrainerVisible(newValue);
        
        firePropertyChange("gridVisible", oldValue, newValue);
        prefs.putBoolean("project.gridVisible", newValue);
    }
    public boolean isGridVisible() {
        return problemDrawingView.getConstrainer() == visibleConstrainer;
    }
    public void setScaleFactor(double newValue) {
        double oldValue = getScaleFactor();
        problemDrawingView.setScaleFactor(newValue);
        solutionDrawingView.setScaleFactor(newValue);
        
        firePropertyChange("scaleFactor", oldValue, newValue);
        prefs.putDouble("project.scaleFactor", newValue);
    }
    public double getScaleFactor() {
        return problemDrawingView.getScaleFactor();
    }
    
    public void clear() {
        ActivityDiagram problemDiagram = new ActivityDiagram();
        problemDiagram.setSimulation(new ActivityModel());
        ActivityDiagram solutionDiagram = new ActivityDiagram();
        solutionDiagram.setSimulation(new ActivityModel());
        
        problemDrawingView.setDrawing(problemDiagram);
        solutionDrawingView.setDrawing(solutionDiagram);
                problemDrawingView.getDrawing().addUndoableEditListener(undo);
                solutionDrawingView.getDrawing().addUndoableEditListener(undo);
    }
    
    public void setArrangement(Arrangeable.Arrangement newValue) {
        if (newValue != arrangement) {
            Arrangeable.Arrangement oldValue = arrangement;
            arrangement = newValue;
            
            mainPanel.removeAll();
            
            switch (arrangement) {
                case HORIZONTAL :
                    splittedProblemPanel.add(problemScrollPane, 
                            labels.getString("problem"));
                    splittedSolutionPanel.add(solutionScrollPane, 
                            labels.getString("solution"));
                    splitPane.setOrientation(JSplitPane.HORIZONTAL_SPLIT);
                    mainPanel.add(splitPane);
                    break;
                case VERTICAL :
                    splittedProblemPanel.add(problemScrollPane, 
                            labels.getString("problem"));
                    splittedSolutionPanel.add(solutionScrollPane, 
                            labels.getString("solution"));
                    splitPane.setOrientation(JSplitPane.VERTICAL_SPLIT);
                    mainPanel.add(splitPane);
                    break;
                case CASCADE :
                    tabbedPane.add(problemScrollPane, 
                            labels.getString("problem"));
                    tabbedPane.add(solutionScrollPane, 
                            labels.getString("solution"));
                    mainPanel.add(tabbedPane);
                    break;
            }
            mainPanel.validate();
            mainPanel.repaint();
            
            prefs.put("project.arrangement", newValue.toString());
            firePropertyChange("arrangement", oldValue, newValue);
        }
    }
    
    public Arrangeable.Arrangement getArrangement() {
        return arrangement;
    }
    
    /**
     * Handles toggle buttons for the tools.
     */
    private class ToolHandler extends ToolAdapter {
        private JToggleButton defaultToolButton;
        public ToolHandler(JToggleButton defaultToolButton) {
            this.defaultToolButton = defaultToolButton;
        }
        
        @Override
        public void toolDone(ToolEvent event) {
            defaultToolButton.setSelected(true);
        }
    }
    
    @Override
    public void markChangesAsSaved() {
        undo.discardAllEdits();
    }
    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {

        mainSplitPane = new javax.swing.JSplitPane();
        mainLeftPanel = new javax.swing.JPanel();
        optionsPanel = new ch.hslu.cm.OptionsPanel();
        mainRightPanel = new javax.swing.JPanel();
        mainPanel = new javax.swing.JPanel();
        tabbedPane = new javax.swing.JTabbedPane();
        problemScrollPane = new javax.swing.JScrollPane();
        problemDrawingView = new org.jhotdraw.draw.DefaultDrawingView();
        solutionScrollPane = new javax.swing.JScrollPane();
        solutionDrawingView = new org.jhotdraw.draw.DefaultDrawingView();

        setLayout(new java.awt.BorderLayout());

        mainLeftPanel.setLayout(new java.awt.BorderLayout());
        mainLeftPanel.add(optionsPanel, java.awt.BorderLayout.CENTER);

        mainSplitPane.setLeftComponent(mainLeftPanel);

        mainRightPanel.setMinimumSize(new java.awt.Dimension(0, 0));
        mainRightPanel.setLayout(new java.awt.BorderLayout());

        mainPanel.setLayout(new java.awt.BorderLayout());

        tabbedPane.setTabPlacement(javax.swing.JTabbedPane.BOTTOM);

        problemScrollPane.setBorder(javax.swing.BorderFactory.createEmptyBorder(0, 0, 0, 0));
        problemScrollPane.setViewportView(problemDrawingView);

        tabbedPane.addTab("null", problemScrollPane);

        solutionScrollPane.setBorder(javax.swing.BorderFactory.createEmptyBorder(0, 0, 0, 0));
        solutionScrollPane.setViewportView(solutionDrawingView);

        tabbedPane.addTab("null", solutionScrollPane);

        tabbedPane.setSelectedIndex(1);

        mainPanel.add(tabbedPane, java.awt.BorderLayout.CENTER);

        mainRightPanel.add(mainPanel, java.awt.BorderLayout.CENTER);

        mainSplitPane.setRightComponent(mainRightPanel);

        add(mainSplitPane, java.awt.BorderLayout.CENTER);
    }// </editor-fold>//GEN-END:initComponents
    
    
    
    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JPanel mainLeftPanel;
    private javax.swing.JPanel mainPanel;
    private javax.swing.JPanel mainRightPanel;
    private javax.swing.JSplitPane mainSplitPane;
    private ch.hslu.cm.OptionsPanel optionsPanel;
    private org.jhotdraw.draw.DefaultDrawingView problemDrawingView;
    private javax.swing.JScrollPane problemScrollPane;
    private org.jhotdraw.draw.DefaultDrawingView solutionDrawingView;
    private javax.swing.JScrollPane solutionScrollPane;
    private javax.swing.JTabbedPane tabbedPane;
    // End of variables declaration//GEN-END:variables
    
}
